#include <linux/module.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <asm/uaccess.h>
#include <linux/ioport.h>
#include <asm/io.h>

//------------------------------
// #DEFINES
//------------------------------

#ifdef _DEBUG
#define PDEBUG(fmt,args...) printk(KERN_DEBUG"parportleds:"fmt, ## args)
#else
# define PDEBUG(fmt,args...) /* rien. pas de dbogage */
#endif

#define TYPE(dev) (MINOR(dev) >>4)
#define NUM(dev) (MINOR(dev) & 0xf

#define PARPORTLEDS_ON 1
#define PARPORTLEDS_OFF 0

#define	PARPORTLEDS_MODULE_NAME "parportleds"
#define PARPORTLEDS_MAJOR 0
#define PARPORTLEDS_BASEIO 0x3bc
#define PARPORTLEDS_BUFFER_SIZE 12

#ifndef SEEK_SET
#define SEEK_SET 0
#endif
#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif
#ifndef SEEK_END
#define SEEK_END 2
#endif

#define HOWMUCH_LEDS 12

//------------------------------
// PARAMETRES

int parm_parportleds_major = PARPORTLEDS_MAJOR;
size_t parm_parportleds_baseio = PARPORTLEDS_BASEIO;
char* parm_parportleds_name = PARPORTLEDS_MODULE_NAME;


static char parportleds_buffer[PARPORTLEDS_BUFFER_SIZE+1] = "000000000000\0";
static struct semaphore parportleds_sem_ecriture; // sert  savoir si le contenu est en cours d'criture

MODULE_PARM(parm_parportleds_major,"i");
MODULE_PARM(parm_parportleds_baseio,"l");
MODULE_PARM(parm_parportleds_name,"s");

//------------------------------


//------------------------------
// Dclarations des fonctions 
// Elles sont static pour viter leur exportation dans le noyau
static int parportleds_release(struct inode* inode, struct file* filp);
static int parportleds_open(struct inode* inode, struct file* filp);
static ssize_t parportleds_read(struct file* filp, char* buff, size_t count, loff_t* offset);
static ssize_t parportleds_write(struct file* filp, const char* buff, size_t count, loff_t* offset);
static void parportleds_strclean(char*);
static loff_t parportleds_llseek (struct file*, loff_t, int);
static void parportleds_bit_set (unsigned char* , int, int );
static void parportleds_set_led(long int, int);
static void parportleds_switch_off_leds();
//


static void parportleds_strclean(char* chaine)
{
		char* c;
		for( c = chaine; c< chaine + PARPORTLEDS_BUFFER_SIZE + 1 ;c++){*c='\0';}
}

static struct file_operations parportleds_fops =
{
		llseek: parportleds_llseek,
		open: parportleds_open,
		release: parportleds_release,
		read : parportleds_read,
		write : parportleds_write
};
//---------------------------------------------------------
int 
parportleds_open(struct inode* inode, struct file* filp)
{
		// incrmente le compteur d'utilisation
    MOD_INC_USE_COUNT;

		// Taille de l'inode = 12
		filp->f_dentry->d_inode->i_size = PARPORTLEDS_BUFFER_SIZE;

    return 0;
}
//---------------------------------------------------------
int 
parportleds_release(struct inode* inode, struct file* filp)
{
		// Dcremente le compteur d'utilisation 
    MOD_DEC_USE_COUNT;

    return 0;
}
//---------------------------------------------------------

static ssize_t parportleds_read(struct file* filp, char* buff, size_t count, loff_t* offset)
{
		int lu;

		// PDEBUG("offset =%li\n",*offset);
		PDEBUG("count =%i\n",count);

		// si la position finale, alors erreur ESPIPE (ref: asm/errno.h)
		if (*offset > PARPORTLEDS_BUFFER_SIZE) 
		{
			printk("<1>parportleds: offset suprieur  taille\n");	
			return -ESPIPE;
		}

		// si fin du fichier => rien  lire
		if (*offset == PARPORTLEDS_BUFFER_SIZE) 
		{
				printk("<1>parportleds: fin du fichier\n");
				return 0;
		}

		// la lecture dpassera la fin du fichier => redimensionnement
		if (*offset+count > PARPORTLEDS_BUFFER_SIZE) 
		{
				count = PARPORTLEDS_BUFFER_SIZE - *offset;
				PDEBUG("redimensionnement. Nouvelle valeur = %i\n",count);
		}

		lu = count-copy_to_user(buff, &parportleds_buffer + *offset ,count);
		PDEBUG("lu =%i\n",lu);

		// Mise  jour de l'offset
		*offset = *offset + lu;
		return (lu);
}
//---------------------------------------------------------
static ssize_t parportleds_write(struct file* filp, const char* buff, size_t count, loff_t* offset)
{
		int ecris, i;

    if (count==0) 
				return 0;

		ecris = count;

 		if (count > PARPORTLEDS_BUFFER_SIZE) 
				ecris = PARPORTLEDS_BUFFER_SIZE;		

		// pose un verrou (dcrmente le smaphore)
		if (down_interruptible(&parportleds_sem_ecriture))
		{
 				return -ERESTARTSYS;
		}

		ecris = ecris - copy_from_user(&parportleds_buffer,buff,ecris);
		parportleds_buffer[ecris] = '\0';

		// mise  jour de l'offset
		*offset = 0;

		// libre le verrou
		up(&parportleds_sem_ecriture);

		PDEBUG("%d octets cris\n",ecris);
		PDEBUG("cris:  %s\n",parportleds_buffer);		

		for (i=1; i<= ecris; i++)
						parportleds_set_led(i, parportleds_buffer[i-1] == '0' ? PARPORTLEDS_ON : PARPORTLEDS_OFF);

		return count;
}
//---------------------------------------------------------
int 
init_module(void)
{

		//*****************************************************
		int result =  register_chrdev(parm_parportleds_major,parm_parportleds_name,&parportleds_fops); 

		if (result<0)
		{
				printk("<1>parportleds: can get major %d\n",parm_parportleds_major);
				return result;
		}

		if (parm_parportleds_major == 0 ) parm_parportleds_major = result;


		//*****************************************************
		if ( check_region(parm_parportleds_baseio,1))
		{
				printk("<1>parportleds: memory already in use\n");
				return -EBUSY;				
		}
		
		request_region(parm_parportleds_baseio,1,"parportleds");

		//*****************************************************
		// Initialisation  'NULL' de la zone mmoire
		//parportleds_strclean(parportleds_buffer);
		
		// Initialisation du smaphore
		sema_init(&parportleds_sem_ecriture,1);

		parportleds_switch_off_leds();

		return 0;
}

//---------------------------------------------------------

void
cleanup_module(void)
{
		unregister_chrdev(parm_parportleds_major,parm_parportleds_name);

		release_region(parm_parportleds_baseio,1);
}

//---------------------------------------------------------

static loff_t parportleds_llseek (struct file* filp, loff_t offset, int origine)
{
		switch (origine) 
		{
				case SEEK_SET:
						break;
				case SEEK_CUR:
						offset += filp->f_pos;
						break;
				case SEEK_END:
						offset += filp->f_dentry->d_inode->i_size;
						break;
				default:
				offset  =-1;						
		}

		if ((offset < 0) || (offset > filp->f_dentry->d_inode->i_size))
				return -EINVAL;
		filp->f_pos = offset;
		return filp->f_pos;
}

//---------------------------------------------------------

static void 
parportleds_bit_set (unsigned char* bits, int pos, int etat)
{
		unsigned char masque;
		int i;
		
		masque = 0x80;
		
		for (i=0; i<(pos % 8); i++)
				masque = masque >> 1;
		
		if (etat)
				bits[pos/8] = bits[pos / 8] | masque;
		else
		{
				bits[pos/8] = bits[pos/8] & (~masque);
		}
		return;
}

//---------------------------------------------------------

static void
parportleds_set_led(long int _numled, int _OnOff)
{
		unsigned char state_byte;
		
		if (_numled < 9)
		{
				state_byte = inb(parm_parportleds_baseio);
				parportleds_bit_set(&state_byte,_numled-1,_OnOff);
				outb(state_byte,parm_parportleds_baseio);
		}
		else
		{
				if (_numled==9) 
				{
						_numled=16;
						_OnOff= _OnOff?0:1 ;
				}
				if (_numled==10) 
				{
						_numled=15;
						_OnOff= _OnOff?0:1 ;
				}
				
				if (_numled==11) _numled=14;
				
				if (_numled==12) 
				{
						_numled=13;
						_OnOff= _OnOff?0:1 ;
				}
				state_byte = inb(parm_parportleds_baseio + 2);
				parportleds_bit_set(&state_byte,_numled-9,_OnOff);
				outb(state_byte,parm_parportleds_baseio + 2);
		}

}

//---------------------------------------------------------

static void parportleds_switch_off_leds(void)
{
		int i;

		for(i=1; i <= HOWMUCH_LEDS ; i++)
		{
				parportleds_set_led(i,PARPORTLEDS_OFF);
		}

}

//---------------------------------------------------------
