#!/bin/bash

# --- backeupe.sh ---
# intrigeri@squat.net
# licence: GPL
# version: 2003 11 30



# U S E R  C O N F I G

RC="$HOME/.backeupe"                      # Config dir
source "${RC}/backeupe.rc"                # Sourcing main config file



# C O N S T A N T S

DATE='/bin/date'                          # Where to find date
TAR='/bin/tar'                            # Where to find tar
GPG='/usr/bin/gpg'                        # Where to find gpg
SPLIT='/usr/bin/split'                    # Where to find split



# F U N C T I O N S

# Function called in case of failure
function fail {
    echo "Failure."
    cd ${OLDDIR}
    exit 1
}


function checks {

    # Misc checks on config files
    if [ ! -d ${RC} ]; then
	echo "Dir ${RC} not found :-("
	fail
    fi

    if [ ! -f ${TARGETS} ]; then
	echo "File ${TARGETS} not found :-("
	fail
    fi
    if [ `wc --bytes ${TARGETS} | awk '{ print $1 }'` == 0 ]; then
	echo "File ${TARGETS} is empty :-("
	fail
    fi

    if [ ! -f ${PARIAS} ]; then
	echo "File ${PARIAS} not found :-("
	fail
    fi
    if [ `wc --bytes ${PARIAS} | awk '{ print $1 }'` == 0 ]; then
	PARIASOPT=
    else
	PARIASOPT="--exclude-from ${PARIAS}"
    fi

    # Backups output dir check
    if [ ! -d ${BACKEUPEDIR} ]; then
	echo "Dir ${BACKEUPEDIR} not found :-("
	fail
    fi

    # Useful binaries check
    if [ ! -x ${DATE} ] ; then
	echo "Program ${DATE} not found :-("
	fail
    fi
    if [ ! -x ${TAR} ] ; then
	echo "Program ${TAR} not found :-("
	fail
    fi
    if [ ! -x ${GPG} ]; then
	echo "Program ${GPG} not found :-("
	fail
    fi
    if [ ! -x ${SPLIT} ]; then
	echo "Program ${SPLIT} not found :-("
	fail
    fi

    # Secret key check
    $GPG --quiet --list-secret-keys ${GPGKEY} > /dev/null
    RES=$?
    if [ ${RES} != 0 ]; then
	echo "The private key ${GPGKEY} is not in your keyring..."
	fail
    fi
}



# T H E   P R O G R A M   I T S E L F   

OLDDIR=`pwd`

if [ x$1 == x ]; then
    PREFIX="${PREFIX}-full"
else
    PREFIX="${PREFIX}-incr"
fi

checks

cd ${BACKEUPEDIR}

NOW=`${DATE} +%Y%m%d-%H%M%S`                  # Current date+time (ex: 20030419-162341)
PREFIX="${PREFIX}-${NOW}"

DOTTGZ="${BACKEUPEDIR}/${PREFIX}.tgz"
DOTGPG="${DOTTGZ}.gpg"


# Taring, compressing, encrypting and splitting

echo "Creating ${DOTGPG}..."

if [ x$1 == x ] ; then

    # 'full' backup
    $TAR \
	--create \
	${PARIASOPT} \
	--label="${PREFIX}" \
	--gzip \
	--verbose \
	--ignore-failed-read \
	--files-from=${TARGETS} \
	2>${BACKEUPEDIR}/${PREFIX}.log \
    | $GPG \
	--encrypt \
	--recipient ${GPGKEY} \
	-z 0 \
    | $SPLIT \
	    --bytes ${MAXFILESIZE} \
	    --suffix-length=2 \
	    - \
	    ${DOTGPG}.
    RES=$?

else

    # 'incremental' backup
    $TAR \
	--create \
	${PARIASOPT} \
	--after-date="$1 hours ago" \
	--label="${PREFIX}" \
	--gzip \
	--verbose \
	--ignore-failed-read \
	--files-from=${TARGETS} \
	2>${BACKEUPEDIR}/${PREFIX}.log \
    | $GPG \
	--encrypt \
	--recipient ${GPGKEY} \
	-z 0 \
    | $SPLIT \
	    --bytes ${MAXFILESIZE} \
	    --suffix-length=2 \
	    - \
	    ${DOTGPG}.
    RES=$?
fi

if [ $RES != 0 ] ; then
    echo "Backup: failed."
    rm ${DOTGPG}*
    fail
else
    echo "Backup: success."
fi


# Signature

echo "Creating ${DOTGPG}.sig..."
$GPG \
    --armor \
    --default-key ${GPGKEY} \
    --output ${DOTGPG}.sig \
    --detach-sign \
    ${DOTGPG}*

RES=$?
if [ $RES != 0 ]; then
    echo "Creation of ${DOTGPG}.sig: failure."
    rm ${PREFIX}*
    fail
else
    echo "Creation of ${DOTGPG}.sig: success."
fi



cd ${OLDDIR}


echo ""
echo "Files created:"

ls \
    --size \
    -1 \
    --color=auto \
    --human-readable \
    ${BACKEUPEDIR}/${PREFIX}*

echo ""
echo "Hopping you'll never need all that... ciao!"
echo ""
